--------------------------------------------------------------------------
--This prefab file is for loading autogenerated Fx prefabs
--------------------------------------------------------------------------

local FxAutogenData = require "prefabs.fx_autogen_data"
local fmodtable = require "defs.sound.fmodtable"
local prefabutil = require "prefabs.prefabutil"
local soundutil = require "util.soundutil"
require "util.kstring"


local function SetupDeathFxFor(inst, target)
	local x, z = target.Transform:GetWorldXZ()
	inst.Transform:SetPosition(x, 0, z)
	inst.Transform:SetRotation(target.Transform:GetFacingRotation() + 180)

	if target.components.coloradder ~= nil then
		target.components.coloradder:AttachChild(inst)
	end
	if target.components.hitstopper ~= nil then
		target.components.hitstopper:AttachChild(inst)
	end

	local function OnSetDir(target, data)
		if data ~= nil and data.attack ~= nil and data.attack:GetDir() ~= nil then
			local dir = DiffAngle(0, data.attack:GetDir())
			if dir < 90 then
				dir = 0
			elseif dir > 90 then
				dir = 180
			else
				dir = target.Transform:GetFacingRotation() + 180
			end
			inst.Transform:SetRotation(dir)
		end
	end
	inst:ListenForEvent("attacked", OnSetDir, target)
	inst:ListenForEvent("knockdown", OnSetDir, target)
	inst:ListenForEvent("knockback", OnSetDir, target)
end

local function OnSetSpawnInstigator(inst, instigator)
	TheTrackers.DebugSpawnEffect(inst, instigator)
	local params = inst.fx_sound_params
	if params then
		if params.window_frames then
			soundutil.PlayWindowedSound(inst, params, instigator)
		else
			-- Only expect one sound per fx, so use a fixed name so we always have a handle.
			inst.sound_handle = soundutil.PlaySoundData(inst, params, "fx_sound", instigator)
		end
		inst.fx_sound_params = nil
	end
end

function MakeAutogenFx(name, params, debug)
	local assets =
	{
		Asset("PKGREF", "scripts/prefabs/autogen/fx/".. name ..".lua"),
		Asset("PKGREF", "scripts/prefabs/fx_autogen.lua"),
		Asset("PKGREF", "scripts/prefabs/fx_autogen_data.lua"),
	}

	-- Fx not loading? Probably because they're spawned from code and not
	-- included in the prefab's dependency list (prefabs in player_side).

	local build = params.build or name
	local bank = params.bank or name
	prefabutil.CollectAssetsForAnim(assets, build, bank, params.bankfile, debug)

	-- this is stripped down from prop_autogen.lua
	local networktype = NetworkType_None
	if params.networked then
		-- If the network type is manually assigned, use that instead of the group setting
		if params.networked == 1 then	-- On
			assert(params.isminimal, "Only minimal network fx_autogen entities are supported")
			if params.isminimal then
				networktype = NetworkType_ClientMinimal
			end
		end -- else it will just be networked OFF, meaning NetworkType_None
	end

	local function fn(prefabname)

		local inst = CreateEntity()
		inst:SetPrefabName(prefabname)

		inst.entity:AddTransform()
		inst.entity:AddAnimState()
		if params.soundevent ~= nil then
			inst.entity:AddSoundEmitter()
		end

		inst:AddTag("FX")
		inst:AddTag("NOCLICK")
		inst.persists = false

		inst.AnimState:SetBank(params.bank or name)
		inst.AnimState:SetBuild(params.build or name)

		if params.anim ~= nil then
			local anim = params.anim
			if params.variations ~= nil then
				anim = anim..tostring(math.random(params.variations))
			end
			-- Uncomment to debug "Could not find anim build"
			--~ TheLog.ch.Prefab:print("PlayAnimation", anim)
			if params.looping then
				inst.AnimState:PlayAnimation(anim, true)
				if params.randomstartframe then
					inst.AnimState:SetFrame(math.random(inst.AnimState:GetCurrentAnimationNumFrames()) - 1)
				end
			else
				inst.AnimState:PlayAnimation(anim)
			end
		end

		if params.shadow then
			inst.AnimState:SetShadowEnabled(true)
		end

		if not params.looping then
			inst:ListenForEvent("animover", inst.Remove)
		end

		if params.scalex ~= nil or params.scaley ~= nil then
			--TODO: ground projected: not a 1-to-1 ratio
			inst.AnimState:SetScale(params.scalex or 1, params.scaley or 1)
		end

		if not params.noflip then
				-- "None", "Flip", "Auto", "Random",
			local sx = params.scalex or 1
			local sy = params.scaley or 1
			if params.flip then
				inst.AnimState:SetScale(-sx, sy)
			elseif not params.randomflip then
				inst.Transform:SetTwoFaced()
			elseif math.random() < .5 then
				inst.AnimState:SetScale(-sx, sy)
			end
		end

		if params.onground then
			inst.AnimState:SetOrientation(ANIM_ORIENTATION.OnGround)
		end

		if params.clip_at_worldedge then
			inst.AnimState:SetClipAtWorldEdge(params.clip_at_worldedge)
		end

		if params.layer ~= nil then
			if params.layer == "bg" then
				inst.AnimState:SetLayer(LAYER_BACKGROUND)
			elseif params.layer == "backdrop" then
				inst.AnimState:SetLayer(LAYER_BACKDROP)
			end
		end

		if params.sortorder ~= nil then
			inst.AnimState:SetSortOrder(params.sortorder)
		end

		if params.finaloffset ~= nil then
			inst.AnimState:SetFinalOffset(params.finaloffset)
		end

		if params.hue ~= nil then
			inst.AnimState:SetHue(params.hue / 360)
		end

		if params.saturation ~= nil then
			inst.AnimState:SetSaturation((params.saturation + 100) / 100)
		end

		if params.brightness ~= nil then
			inst.AnimState:SetBrightness((params.brightness + 100) / 100)
		end

		inst:AddComponent("colormultiplier")
		if params.multcolor ~= nil then
			inst.components.colormultiplier:PushColor("fx_autogen", HexToRGBFloats(StrToHex(params.multcolor)))
		end

		inst:AddComponent("coloradder")
		if params.addcolor ~= nil then
			inst.components.coloradder:PushColor("fx_autogen", HexToRGBFloats(StrToHex(params.addcolor)))
		end

		inst:AddComponent("bloomer")
		if params.bloom ~= nil then
			if params.glowcolor ~= nil then
				local r, g, b = HexToRGBFloats(StrToHex(params.glowcolor))
				inst.components.bloomer:PushBloom("fx_autogen", r, g, b, params.bloom / 100)
			else
				inst.components.bloomer:PushBloom("fx_autogen", params.bloom / 100)
			end
		end

		if params.lightoverride ~= nil then
			inst.AnimState:SetLightOverride(params.lightoverride / 100)
		end

		if params.additive ~= nil then
			inst.AnimState:SetBlendMode(BlendMode.id.PremultipliedAdditive)
		end

		inst:AddComponent("hitstopper")

		if params.timelines then
			inst:AddComponent("fxtimeline", params.timelines)
		end

		if params.target_tint ~= nil then
			inst.target_tint = HexToRGB(StrToHex(params.target_tint))
		end

		if params.soundevent ~= nil then
			-- See FxEditor to see all fx with missing sound, but assert to make missing sound obvious to all.
			dbassert(fmodtable.Event[params.soundevent], params.soundevent)
			inst.fx_sound_params = params
		end

		inst.SetupDeathFxFor = SetupDeathFxFor

		inst.OnSetSpawnInstigator = OnSetSpawnInstigator

		-- Add this tag to disable history on this FX, for performance reasons.
		if params.disabledebughistory then
			inst:AddTag("dbg_nohistory")
		end

		return inst
	end

	return Prefab(name, fn, assets, nil, nil, networktype)
end

local ret = {}
local target_groups = {}

for name, params in pairs(FxAutogenData) do
	ret[#ret + 1] = MakeAutogenFx(name, params)
	if params.target_prefab
		and not (params.group or ""):lower():startswith("test")
	then
		local g = target_groups[params.target_prefab] or {}
		table.insert(g, name)
		target_groups[params.target_prefab] = g
	end
end
for prefab,g in pairs(target_groups) do
	table.insert(ret, Prefab(GroupPrefab("fx_for_".. prefab), nil, nil, g))
end

local _, group_prefabs = prefabutil.CreateGroupPrefabs(FxAutogenData, ret)
table.insert(ret, Prefab(GroupPrefab("all_fx_groups"), nil, nil, group_prefabs))

return table.unpack(ret)
